/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/
 */
package com.xpn.xwiki.plugin.skinx;

import org.xwiki.extension.distribution.internal.DistributionManager;

import java.util.Collections;
import java.util.Set;

import com.xpn.xwiki.XWikiContext;
import com.xpn.xwiki.XWikiException;
import com.xpn.xwiki.web.Utils;

/**
 * Skin Extension plugin to use extension files from JAR resources.
 *
 * @version $Id: 75f450dc803e0868df7c57610db93c093d2193de $
 */
public abstract class AbstractResourceSkinExtensionPlugin extends AbstractSkinExtensionPlugin
{
    /**
     * XWiki plugin constructor.
     *
     * @param name The name of the plugin, which can be used for retrieving the plugin API from velocity. Unused.
     * @param className The canonical classname of the plugin. Unused.
     * @param context The current request context.
     * @see com.xpn.xwiki.plugin.XWikiDefaultPlugin#XWikiDefaultPlugin(String,String,com.xpn.xwiki.XWikiContext)
     */
    public AbstractResourceSkinExtensionPlugin(String name, String className, XWikiContext context)
    {
        super(name, className, context);
    }

    /**
     * Get the action which the url should specify for calling this resource.
     *
     * @return String Action name.
     */
    protected abstract String getAction();

    /**
     * Takes a URL string and outputs a link which will cause the browser to load the url.
     *
     * @param url String representation of the url to load (eg: {@code /res/url.js})
     * @param resourceName name of the pulled resource
     * @param context the current request context
     * @return HTML code linking to the pulled resource (eg: {@code <script type="text/javascript" src="/res/url.js"/>})
     */
    protected abstract String generateLink(String url, String resourceName, XWikiContext context);

    @Override
    public String getLink(String resourceName, XWikiContext context)
    {
        // If the current user has access to Main.WebHome, we will use this document in the URL
        // to serve the resource. This way, the resource can be efficiently cached, since it has a
        // common URL for any page.
        // Otherwise, we link to the current document, which means that the extension will be accessible if the current
        // document is accessible as well.
        try {
            String page = context.getWiki().getDefaultSpace(context) + "." + context.getWiki().getDefaultPage(context);
            if (!context.getWiki().getRightService().hasAccessLevel("view", context.getUser(), page, context)) {
                page = context.getDoc().getFullName();
            }
            String version = sanitize(Utils.getComponent(
                DistributionManager.class).getDistributionExtension().getId().getVersion().getValue());
            return generateLink(context.getWiki().getURL(page, getAction(),
                "resource=" + sanitize(resourceName) + "&amp;v=" + version
                    + parametersAsQueryString(resourceName, context), context), resourceName, context);
        } catch (XWikiException e) {
            // Do nothing here; we can't access the wiki, so don't link to this resource at all.
            return "";
        }
    }

    /**
     * {@inheritDoc}
     * <p>
     * There is no support for always used resource-based extensions yet.
     * </p>
     *
     * @see AbstractSkinExtensionPlugin#getAlwaysUsedExtensions(XWikiContext)
     */
    @Override
    public Set<String> getAlwaysUsedExtensions(XWikiContext context)
    {
        // There is no mean to define an always used extension for something else than a document extension now,
        // so for resources-based extensions, we return an emtpy set.
        // An idea for the future could be to have an API for plugins and components to register always used resources
        // extensions.
        return Collections.emptySet();
    }

    /**
     * {@inheritDoc}
     * <p>
     * Not supported for resource extensions.
     * </p>
     *
     * @see com.xpn.xwiki.plugin.skinx.AbstractSkinExtensionPlugin#hasPageExtensions(com.xpn.xwiki.XWikiContext)
     */
    @Override
    public boolean hasPageExtensions(XWikiContext context)
    {
        return false;
    }
}
