
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






/**
 * NullableVar16Char implements a vector of values which could be null.  Elements in the vector
 * are first checked against a fixed length vector of boolean values.  Then the element is retrieved
 * from the base class (if not null).
 *
 * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
 */
@SuppressWarnings("unused")
public final class NullableVar16CharVector extends BaseValueVector implements VariableWidthVector, NullableVector{

  private int valueCount;
  final UInt1Vector bits;
  final Var16CharVector values;
  private final Accessor accessor;
  private final Mutator mutator;

  public NullableVar16CharVector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
    this.bits = new UInt1Vector(null, allocator);
    this.values = new Var16CharVector(field, allocator);
    this.accessor = new Accessor();
    this.mutator = new Mutator();
  }
  
  public int getValueCapacity(){
    return Math.min(bits.getValueCapacity(), values.getValueCapacity());
  }

  public int getCurrentValueCount() {
    return values.getCurrentValueCount();
  }

  public void setCurrentValueCount(int count) {
    values.setCurrentValueCount(count);
  }

  @Override
  public DrillBuf[] getBuffers(boolean clear) {
    DrillBuf[] buffers = ObjectArrays.concat(bits.getBuffers(clear), values.getBuffers(clear), DrillBuf.class);
    if (clear) {
      clear();
    }
    return buffers;
  }
  
  @Override
  public void clear() {
    valueCount = 0;
    bits.clear();
    values.clear();
  }
  
  public int getBufferSize(){
    return values.getBufferSize() + bits.getBufferSize();
  }

  public DrillBuf getData(){
    return values.getData();
  }

  public Var16CharVector getValuesVector() {
    return values;
  }

  @Override
  public SerializedField getMetadata() {
    return getMetadataBuilder()
             .setValueCount(valueCount)
             .setVarByteLength(values.getVarByteLength())
             .setBufferLength(getBufferSize())
             .build();
  }

  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryRuntimeException("Failure while allocating buffer.");
    }
  }
  
  @Override
  public boolean allocateNewSafe() {
    if(!values.allocateNewSafe()) return false;
    if(!bits.allocateNewSafe()) return false;
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
    return true;
  }

  @Override
  public void allocateNew(int totalBytes, int valueCount) {
    values.allocateNew(totalBytes, valueCount);
    bits.allocateNew(valueCount);
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
  }

  @Override
  public int load(int dataBytes, int valueCount, DrillBuf buf){
    clear();
    this.valueCount = valueCount;
    int loaded = bits.load(valueCount, buf);
    
    // remove bits part of buffer.
    buf = buf.slice(loaded, buf.capacity() - loaded);
    dataBytes -= loaded;
    loaded += values.load(dataBytes, valueCount, buf);
    this.mutator.lastSet = valueCount;
    return loaded;
  }
  
  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    assert this.field.matches(metadata) : String.format("The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int loaded = load(metadata.getBufferLength(), metadata.getValueCount(), buffer);
    assert metadata.getBufferLength() == loaded : String.format("Expected to load %d bytes but actually loaded %d bytes", metadata.getBufferLength(), loaded);
  }
  
  @Override
  public int getByteCapacity(){
    return values.getByteCapacity();
  }

  @Override
  public int getCurrentSizeInBytes(){
    return values.getCurrentSizeInBytes();
  }

  
  public TransferPair getTransferPair(){
    return new TransferImpl(getField());
  }
  public TransferPair getTransferPair(FieldReference ref){
    return new TransferImpl(getField().clone(ref));
  }

  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((NullableVar16CharVector) to);
  }

  
  public void transferTo(NullableVar16CharVector target){
    bits.transferTo(target.bits);
    values.transferTo(target.values);
    target.valueCount = valueCount;
    target.mutator.lastSet = mutator.lastSet;
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, NullableVar16CharVector target) {
    bits.splitAndTransferTo(startIndex, length, target.bits);
    values.splitAndTransferTo(startIndex, length, target.values);
    target.mutator.lastSet = length - 1;
  }
  
  private class TransferImpl implements TransferPair{
    NullableVar16CharVector to;
    
    public TransferImpl(MaterializedField field){
      this.to = new NullableVar16CharVector(field, allocator);
    }

    public TransferImpl(NullableVar16CharVector to){
      this.to = to;
    }

    public NullableVar16CharVector getTo(){
      return to;
    }
    
    public void transfer(){
      transferTo(to);
    }

    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public boolean copyValueSafe(int fromIndex, int toIndex) {
      return to.copyFromSafe(fromIndex, toIndex, NullableVar16CharVector.this);
    }
  }
  
  public Accessor getAccessor(){
    return accessor;
  }
  
  public Mutator getMutator(){
    return mutator;
  }
  
  public Var16CharVector convertToRequiredVector(){
    Var16CharVector v = new Var16CharVector(getField().getOtherNullableVersion(), allocator);
    v.data = values.data;
    v.valueCount = this.valueCount;
    v.data.retain();
    clear();
    return v;
  }

  
  protected void copyFrom(int fromIndex, int thisIndex, NullableVar16CharVector from){
    if (!from.getAccessor().isNull(fromIndex)) {
      mutator.set(thisIndex, from.getAccessor().get(fromIndex));
    }
  }

  
  public boolean copyFromSafe(int fromIndex, int thisIndex, Var16CharVector from){
    if(!mutator.fillEmpties(thisIndex)) return false;
    boolean success = values.copyFromSafe(fromIndex, thisIndex, from);
    success = success && bits.getMutator().setSafe(thisIndex, 1);
    return success;    
  }
  
  public boolean copyFromSafe(int fromIndex, int thisIndex, NullableVar16CharVector from){
    if(!mutator.fillEmpties(thisIndex)) return false;
    boolean b1 = bits.copyFromSafe(fromIndex, thisIndex, from.bits);
    boolean b2 = values.copyFromSafe(fromIndex, thisIndex, from.values);
    return b1 && b2;
  }

  public long getDataAddr(){
    return values.getDataAddr();
  }
  
  public long getBitAddr(){
    return bits.getDataAddr();
  }
  
  public long getOffsetAddr(){
    return values.getOffsetAddr();
  }
  
  public final class Accessor implements ValueVector.Accessor, VariableWidthVector.VariableWidthAccessor{

    final FieldReader reader = new NullableVar16CharReaderImpl(NullableVar16CharVector.this);
    final UInt1Vector.Accessor bAccessor = bits.getAccessor();
    final Var16CharVector.Accessor vAccessor = values.getAccessor();
    
    public FieldReader getReader(){
      return reader;
    }
    
    /**
     * Get the element at the specified position.
     *
     * @param   index   position of the value
     * @return  value of the element, if not null
     * @throws  NullValueException if the value is null
     */
    public byte[] get(int index) {
      assert !isNull(index) : "Tried to get null value";
      return vAccessor.get(index);
    }

    public boolean isNull(int index) {
      return isSet(index) == 0;
    }

    public int isSet(int index){
      return bAccessor.get(index);
    }
    
    public long getStartEnd(int index){
      return vAccessor.getStartEnd(index);
    }
    
    public long getOffsetAddr(){
      return values.getOffsetAddr();
    }
    

    


    public int getValueLength(int index) {
      return values.getAccessor().getValueLength(index);
    }

    public void get(int index, NullableVar16CharHolder holder){
      vAccessor.get(index, holder);
      holder.isSet = bAccessor.get(index);

    }

    @Override
    public String getObject(int index) {
      if (isNull(index)) {
          return null;
      }else{
        return vAccessor.getObject(index);
      }
    }


    public int getValueCount(){
      return valueCount;
    }
    
    public void reset(){}
  }
  
  public final class Mutator implements ValueVector.Mutator, NullableVectorDefinitionSetter, VariableWidthVector.VariableWidthMutator {
    
    private int setCount;
     private int lastSet = -1;

    private Mutator(){
    }

    public Var16CharVector getVectorWithValues(){
      return values;
    }

    public void setIndexDefined(int index){
      bits.getMutator().set(index, 1);
    }

    /**
     * Set the variable length element at the specified index to the supplied byte array.
     *
     * @param index   position of the bit to set
     * @param bytes   array of bytes to write
     */
    public void set(int index, byte[] value) {
      setCount++;
      for (int i = lastSet + 1; i < index; i++) {
        values.getMutator().set(i, new byte[]{});
      }
      bits.getMutator().set(index, 1);
      values.getMutator().set(index, value);
      lastSet = index;
    }

    private boolean fillEmpties(int index){
      for (int i = lastSet + 1; i < index; i++) {
        if(!values.getMutator().setSafe(i, new byte[]{})) return false;
      }
      lastSet = index;

      return true;
    }

    public boolean setValueLengthSafe(int index, int length) {
      return values.getMutator().setValueLengthSafe(index, length);
    }
    
    public boolean setSafe(int index, byte[] value, int start, int length) {
      if(!fillEmpties(index)) return false;

      boolean b1 = bits.getMutator().setSafe(index, 1);
      boolean b2 = values.getMutator().setSafe(index, value, start, length);
      if(b1 && b2){
        setCount++;
        lastSet = index;
        return true;
      }else{
        return false;
      }
    }

    public void setSkipNull(int index, Var16CharHolder holder){
      values.getMutator().set(index, holder);
    }

    public void setSkipNull(int index, NullableVar16CharHolder holder){
      values.getMutator().set(index, holder);
    }
    
    
    public void set(int index, NullableVar16CharHolder holder){
      for (int i = lastSet + 1; i < index; i++) {
        values.getMutator().set(i, new byte[]{});
      }
      bits.getMutator().set(index, holder.isSet);
      values.getMutator().set(index, holder);
      lastSet = index;
    }

    public void set(int index, Var16CharHolder holder){
      for (int i = lastSet + 1; i < index; i++) {
        values.getMutator().set(i, new byte[]{});
      }
      bits.getMutator().set(index, 1);
      values.getMutator().set(index, holder);
      lastSet = index;
    }
    
    public boolean isSafe(int outIndex) {
      return outIndex < NullableVar16CharVector.this.getValueCapacity();
    }

    //public boolean setSafe(int index, int isSetNullableVar16CharHolder  value){

    public void set(int index, int isSet, int startField, int endField, DrillBuf bufferField ){
      for (int i = lastSet + 1; i < index; i++) {
        values.getMutator().set(i, new byte[]{});
      }
      bits.getMutator().set(index, isSet);
      values.getMutator().set(index, startField, endField, bufferField);
      lastSet = index;
    }
    
    public boolean setSafe(int index, int isSet, int startField, int endField, DrillBuf bufferField ) {
      if(!fillEmpties(index)) return false;
      
      boolean b1 = bits.getMutator().setSafe(index, isSet);
      boolean b2 = values.getMutator().setSafe(index, startField, endField, bufferField);
      if(b1 && b2){
        setCount++;
        lastSet = index;
        return true;
      }else{
        return false;
      }

    }


    public boolean setSafe(int index, NullableVar16CharHolder value) {

      if(!fillEmpties(index)) return false;
      boolean b1 = bits.getMutator().setSafe(index, value.isSet);
      boolean b2 = values.getMutator().setSafe(index, value);
      if(b1 && b2){
        setCount++;
        lastSet = index;
        return true;
      }else{
        return false;
      }

    }

    public boolean setSafe(int index, Var16CharHolder value) {

      if(!fillEmpties(index)) return false;
      boolean b1 = bits.getMutator().setSafe(index, 1);
      boolean b2 = values.getMutator().setSafe(index, value);
      if(b1 && b2){
        setCount++;
        lastSet = index;
        return true;
      }else{
        return false;
      }

    }
    

    public void setValueCount(int valueCount) {
      assert valueCount >= 0;
      fillEmpties(valueCount);
      NullableVar16CharVector.this.valueCount = valueCount;
      values.getMutator().setValueCount(valueCount);
      bits.getMutator().setValueCount(valueCount);
    }
    
    public boolean noNulls(){
      return valueCount == setCount;
    }
    
    public void generateTestData(int valueCount){
      bits.getMutator().generateTestDataAlt(valueCount);
      values.getMutator().generateTestData(valueCount);
      lastSet = valueCount;
      setValueCount(valueCount);
    }
    
    public void reset(){
      setCount = 0;
      lastSet = -1;
    }
    
  }
}


