/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






/**
 * Decimal28Dense implements a vector of fixed width values.  Elements in the vector are accessed
 * by position, starting from the logical start of the vector.  Values should be pushed onto the
 * vector sequentially, but may be randomly accessed.
 *   The width of each element is 12 byte(s)
 *   The equivalent Java primitive is 'DrillBuf'
 *
 * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
 */
@SuppressWarnings("unused")
public final class Decimal28DenseVector extends BaseDataValueVector implements FixedWidthVector{

  private final Accessor accessor = new Accessor();
  private final Mutator mutator = new Mutator();

  private int allocationValueCount = INITIAL_VALUE_ALLOCATION;
  private int allocationMonitor = 0;
  
  public Decimal28DenseVector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
  }

  public int getValueCapacity(){
    return (int) (data.capacity() *1.0 / 12);
  }

  public Accessor getAccessor(){
    return accessor;
  }
  
  public Mutator getMutator(){
    return mutator;
  }


  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryRuntimeException("Failure while allocating buffer.");
    }
  }
  
  public boolean allocateNewSafe() {
    clear();
    if (allocationMonitor > 10) {
      allocationValueCount = Math.max(8, (int) (allocationValueCount / 2));
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      allocationValueCount = (int) (allocationValueCount * 2);
      allocationMonitor = 0;
    }
    this.data = allocator.buffer(allocationValueCount * 12);
    if(data == null) return false;
    this.data.readerIndex(0);
    return true;
  }

  /**
   * Allocate a new buffer that supports setting at least the provided number of values.  May actually be sized bigger depending on underlying buffer rounding size. Must be called prior to using the ValueVector.
   * @param valueCount
   */
  public void allocateNew(int valueCount) {
    clear();
    this.data = allocator.buffer(valueCount * 12);
    this.data.readerIndex(0);
    this.allocationValueCount = valueCount;
  }

  /**
   * {@inheritDoc}
   */
  public void zeroVector() {
    data.setZero(0, data.capacity());
  }

  @Override
  public SerializedField getMetadata() {
    return getMetadataBuilder()
             .setValueCount(valueCount)
             .setBufferLength(getBufferSize())
             .build();
  }

  public int getBufferSize() {
    if(valueCount == 0) return 0;
    return valueCount * 12;
  }

  @Override
  public int load(int valueCount, DrillBuf buf){
    clear();
    this.valueCount = valueCount;
    int len = valueCount * 12;
    data = buf.slice(0, len);
    data.retain();
    data.writerIndex(len);
    return len;
  }
  
  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    assert this.field.matches(metadata) : String.format("The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int loaded = load(metadata.getValueCount(), buffer);
    assert metadata.getBufferLength() == loaded : String.format("Expected to load %d bytes but actually loaded %d bytes", metadata.getBufferLength(), loaded);
  }
  
  public TransferPair getTransferPair(){
    return new TransferImpl(getField());
  }
  public TransferPair getTransferPair(FieldReference ref){
    return new TransferImpl(getField().clone(ref));
  }

  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((Decimal28DenseVector) to);
  }

  public void transferTo(Decimal28DenseVector target){
    target.data = data;
    target.data.retain();
    target.data.writerIndex(data.writerIndex());
    target.valueCount = valueCount;
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, Decimal28DenseVector target) {
    int currentWriterIndex = data.writerIndex();
    int startPoint = startIndex * 12;
    int sliceLength = length * 12;
    target.valueCount = length;
    target.data = this.data.slice(startPoint, sliceLength);
    target.data.writerIndex(sliceLength);
    target.data.retain();
  }
  
  private class TransferImpl implements TransferPair{
    Decimal28DenseVector to;
    
    public TransferImpl(MaterializedField field){
      this.to = new Decimal28DenseVector(field, allocator);
    }

    public TransferImpl(Decimal28DenseVector to) {
      this.to = to;
    }
    
    public Decimal28DenseVector getTo(){
      return to;
    }
    
    public void transfer(){
      transferTo(to);
    }

    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }
    
    @Override
    public boolean copyValueSafe(int fromIndex, int toIndex) {
      return to.copyFromSafe(fromIndex, toIndex, Decimal28DenseVector.this);
    }
  }
  
  protected void copyFrom(int fromIndex, int thisIndex, Decimal28DenseVector from){
    from.data.getBytes(fromIndex * 12, data, thisIndex * 12, 12);
     
  }
  
  public boolean copyFromSafe(int fromIndex, int thisIndex, Decimal28DenseVector from){
    if(thisIndex >= getValueCapacity()) {
      decrementAllocationMonitor();
      return false;
    }
    copyFrom(fromIndex, thisIndex, from);
    return true;
  }

  private void decrementAllocationMonitor() {
    if (allocationMonitor > 0) {
      allocationMonitor = 0;
    }
    --allocationMonitor;
  }

  private void incrementAllocationMonitor() {
    ++allocationMonitor;
  }

  public final class Accessor extends BaseValueVector.BaseAccessor{

    final FieldReader reader = new Decimal28DenseReaderImpl(Decimal28DenseVector.this);
    
    public FieldReader getReader(){
      return reader;
    }
    
    public int getValueCount() {
      return valueCount;
    }
    
    public boolean isNull(int index){
      return false;
    }
    

    public DrillBuf get(int index) {
      return data.slice(index * 12, 12);
    }


    public void get(int index, Decimal28DenseHolder holder) {

        holder.start = index * 12;

        holder.buffer = data;

        /* The buffer within the value vector is little endian.
         * For the dense representation though, we use big endian
         * byte ordering (internally). This is because we shift bits to the right and
         * big endian ordering makes sense for this purpose.  So we have to deal with
         * the sign bit for the two representation in a slightly different fashion
         */
        holder.scale = getField().getScale();
        holder.precision = getField().getPrecision();
    }

    public void get(int index, NullableDecimal28DenseHolder holder) {
        holder.isSet = 1;
        holder.start = index * 12;

        holder.buffer = data;
    }

      @Override
      public BigDecimal getObject(int index) {
      return org.apache.drill.exec.util.DecimalUtility.getBigDecimalFromDense(data, index * 12, 3, getField().getScale(), 28, 12);
    }

     
 }
 
 /**
  * Decimal28Dense.Mutator implements a mutable vector of fixed width values.  Elements in the
  * vector are accessed by position from the logical start of the vector.  Values should be pushed
  * onto the vector sequentially, but may be randomly accessed.
  *   The width of each element is 12 byte(s)
  *   The equivalent Java primitive is 'DrillBuf'
  *
  * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
  */
  public final class Mutator extends BaseValueVector.BaseMutator{

    private Mutator(){};
   /**
    * Set the element at the given index to the given value.  Note that widths smaller than
    * 32 bits are handled by the DrillBuf interface.
    *
    * @param index   position of the bit to set
    * @param value   value to set
    */
   public void set(int index, DrillBuf value) {
     data.setBytes(index * 12, value, 0, 12);
   }

   public boolean setSafe(int index, DrillBuf value) {
     if(index >= getValueCapacity()) {
       decrementAllocationMonitor();
       return false;
     }
     data.setBytes(index * 12, value, 0, 12);
     return true;
   }


   public void set(int index, Decimal28DenseHolder holder){
     set(index, holder.start, holder.buffer);
   }

   void set(int index, NullableDecimal28DenseHolder holder){
     set(index, holder.start, holder.buffer);
   }
   
   public boolean setSafe(int index,  NullableDecimal28DenseHolder holder){
     return setSafe(index, holder.start, holder.buffer);
   }
   public boolean setSafe(int index,  Decimal28DenseHolder holder){
     return setSafe(index, holder.start, holder.buffer);
   }
   
   public boolean setSafe(int index, int start, DrillBuf buffer){
     if(index >= getValueCapacity()) {
       decrementAllocationMonitor();
       return false;
     }
     set(index, start, buffer);
     return true;
   }
   
   public void set(int index, int start, DrillBuf buffer){
     data.setBytes(index * 12, buffer, start, 12);
   }


   @Override
   public void generateTestData(int count) {
     setValueCount(count);
     boolean even = true;
     for(int i =0; i < valueCount; i++, even = !even){
       byte b = even ? Byte.MIN_VALUE : Byte.MAX_VALUE;
       for(int w = 0; w < 12; w++){
         data.setByte(i + w, b);
       }
     }
   }
   
   


    
  

  
   public void setValueCount(int valueCount) {
     int currentValueCapacity = getValueCapacity();
     Decimal28DenseVector.this.valueCount = valueCount;
     int idx = (12 * valueCount);
     if (valueCount > 0 && currentValueCapacity > valueCount * 2) {
       incrementAllocationMonitor();
     } else if (allocationMonitor > 0) {
       allocationMonitor = 0;
     }
     VectorTrimmer.trim(data, idx);
   }




  
 }
}

 

