/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.record;

import com.google.common.collect.Lists;

import org.apache.drill.common.types.TypeProtos;
import org.apache.drill.exec.expr.TypeHelper;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.vector.*;

/** Wrapper around VectorAccessible to iterate over the records and fetch fields within a record. */
public class RecordValueAccessor {
  static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(RecordValueAccessor.class);

  private VectorAccessible batch;
  private int currentIndex;
  private ValueVector[] vectors;

  public RecordValueAccessor(VectorAccessible batch) {
    this.batch = batch;

    resetIterator();
    initVectorWrappers();
  }

  public void resetIterator() {
    currentIndex = -1;
  }

  private void initVectorWrappers() {
    BatchSchema schema = batch.getSchema();
    vectors = new ValueVector[schema.getFieldCount()];
    int fieldId = 0;
    for (MaterializedField field : schema) {
      Class<?> vectorClass = TypeHelper.getValueVectorClass(field.getType().getMinorType(), field.getType().getMode());
      vectors[fieldId] = batch.getValueAccessorById(vectorClass, fieldId).getValueVector();
      fieldId++;
    }
  }

  public boolean next() {
    return ++currentIndex < batch.getRecordCount();
  }

  public void getFieldById(int fieldId, ComplexHolder holder) {
    holder.isSet = vectors[fieldId].getAccessor().isNull(currentIndex) ? 1 : 0;
    holder.reader = (vectors[fieldId]).getAccessor().getReader();
    holder.reader.setPosition(currentIndex);
  }

  public void getFieldById(int fieldId, NullableTinyIntHolder holder){
    ((NullableTinyIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, TinyIntHolder holder){
    ((TinyIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedTinyIntHolder holder){
    ((RepeatedTinyIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableUInt1Holder holder){
    ((NullableUInt1Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, UInt1Holder holder){
    ((UInt1Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedUInt1Holder holder){
    ((RepeatedUInt1Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableUInt2Holder holder){
    ((NullableUInt2Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, UInt2Holder holder){
    ((UInt2Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedUInt2Holder holder){
    ((RepeatedUInt2Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableSmallIntHolder holder){
    ((NullableSmallIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, SmallIntHolder holder){
    ((SmallIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedSmallIntHolder holder){
    ((RepeatedSmallIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableIntHolder holder){
    ((NullableIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, IntHolder holder){
    ((IntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedIntHolder holder){
    ((RepeatedIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableUInt4Holder holder){
    ((NullableUInt4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, UInt4Holder holder){
    ((UInt4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedUInt4Holder holder){
    ((RepeatedUInt4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableFloat4Holder holder){
    ((NullableFloat4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Float4Holder holder){
    ((Float4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedFloat4Holder holder){
    ((RepeatedFloat4Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableTimeHolder holder){
    ((NullableTimeVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, TimeHolder holder){
    ((TimeVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedTimeHolder holder){
    ((RepeatedTimeVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableIntervalYearHolder holder){
    ((NullableIntervalYearVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, IntervalYearHolder holder){
    ((IntervalYearVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedIntervalYearHolder holder){
    ((RepeatedIntervalYearVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal9Holder holder){
    ((NullableDecimal9Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal9Holder holder){
    ((Decimal9Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal9Holder holder){
    ((RepeatedDecimal9Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableBigIntHolder holder){
    ((NullableBigIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, BigIntHolder holder){
    ((BigIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedBigIntHolder holder){
    ((RepeatedBigIntVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableUInt8Holder holder){
    ((NullableUInt8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, UInt8Holder holder){
    ((UInt8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedUInt8Holder holder){
    ((RepeatedUInt8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableFloat8Holder holder){
    ((NullableFloat8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Float8Holder holder){
    ((Float8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedFloat8Holder holder){
    ((RepeatedFloat8Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDateHolder holder){
    ((NullableDateVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, DateHolder holder){
    ((DateVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDateHolder holder){
    ((RepeatedDateVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableTimeStampHolder holder){
    ((NullableTimeStampVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, TimeStampHolder holder){
    ((TimeStampVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedTimeStampHolder holder){
    ((RepeatedTimeStampVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal18Holder holder){
    ((NullableDecimal18Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal18Holder holder){
    ((Decimal18Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal18Holder holder){
    ((RepeatedDecimal18Vector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableTimeStampTZHolder holder){
    ((NullableTimeStampTZVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, TimeStampTZHolder holder){
    ((TimeStampTZVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedTimeStampTZHolder holder){
    ((RepeatedTimeStampTZVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableIntervalDayHolder holder){
    ((NullableIntervalDayVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, IntervalDayHolder holder){
    ((IntervalDayVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedIntervalDayHolder holder){
    ((RepeatedIntervalDayVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableIntervalHolder holder){
    ((NullableIntervalVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, IntervalHolder holder){
    ((IntervalVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedIntervalHolder holder){
    ((RepeatedIntervalVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal28DenseHolder holder){
    ((NullableDecimal28DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal28DenseHolder holder){
    ((Decimal28DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal28DenseHolder holder){
    ((RepeatedDecimal28DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal38DenseHolder holder){
    ((NullableDecimal38DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal38DenseHolder holder){
    ((Decimal38DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal38DenseHolder holder){
    ((RepeatedDecimal38DenseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal38SparseHolder holder){
    ((NullableDecimal38SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal38SparseHolder holder){
    ((Decimal38SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal38SparseHolder holder){
    ((RepeatedDecimal38SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableDecimal28SparseHolder holder){
    ((NullableDecimal28SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Decimal28SparseHolder holder){
    ((Decimal28SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedDecimal28SparseHolder holder){
    ((RepeatedDecimal28SparseVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableVarBinaryHolder holder){
    ((NullableVarBinaryVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, VarBinaryHolder holder){
    ((VarBinaryVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedVarBinaryHolder holder){
    ((RepeatedVarBinaryVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableVarCharHolder holder){
    ((NullableVarCharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, VarCharHolder holder){
    ((VarCharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedVarCharHolder holder){
    ((RepeatedVarCharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableVar16CharHolder holder){
    ((NullableVar16CharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, Var16CharHolder holder){
    ((Var16CharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedVar16CharHolder holder){
    ((RepeatedVar16CharVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, NullableBitHolder holder){
    ((NullableBitVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, BitHolder holder){
    ((BitVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

  public void getFieldById(int fieldId, RepeatedBitHolder holder){
    ((RepeatedBitVector) vectors[fieldId]).getAccessor().get(currentIndex, holder);
  }

}
