
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.expr;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;





import org.apache.drill.common.expression.SchemaPath;
import org.apache.drill.common.types.TypeProtos.MajorType;
import org.apache.drill.exec.record.MaterializedField;
import org.apache.drill.exec.vector.accessor.*;
import org.apache.drill.exec.vector.complex.RepeatedMapVector;

public class TypeHelper {
  static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(TypeHelper.class);

  private static final int WIDTH_ESTIMATE = 50;

  public static int getSize(MajorType major) {
    switch (major.getMinorType()) {
    case TINYINT:
      return 1;
    case UINT1:
      return 1;
    case UINT2:
      return 2;
    case SMALLINT:
      return 2;
    case INT:
      return 4;
    case UINT4:
      return 4;
    case FLOAT4:
      return 4;
    case TIME:
      return 4;
    case INTERVALYEAR:
      return 4;
    case DECIMAL9:
      return 4;
    case BIGINT:
      return 8;
    case UINT8:
      return 8;
    case FLOAT8:
      return 8;
    case DATE:
      return 8;
    case TIMESTAMP:
      return 8;
    case DECIMAL18:
      return 8;
    case TIMESTAMPTZ:
      return 12;
    case INTERVALDAY:
      return 12;
    case INTERVAL:
      return 16;
    case DECIMAL28DENSE:
      return 12;
    case DECIMAL38DENSE:
      return 16;
    case DECIMAL38SPARSE:
      return 24;
    case DECIMAL28SPARSE:
      return 20;
    case VARBINARY:
      return 4 + WIDTH_ESTIMATE;
    case VARCHAR:
      return 4 + WIDTH_ESTIMATE;
    case VAR16CHAR:
      return 4 + WIDTH_ESTIMATE;
    case BIT:
      return 1;
      case FIXEDCHAR: return major.getWidth();
      case FIXED16CHAR: return major.getWidth();
      case FIXEDBINARY: return major.getWidth();
    }
    throw new UnsupportedOperationException();
  }

  public static SqlAccessor getSqlAccessor(ValueVector vector){
    switch(vector.getField().getType().getMinorType()){
    case TINYINT:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new TinyIntAccessor((TinyIntVector) vector);
        case OPTIONAL:
          return new NullableTinyIntAccessor((NullableTinyIntVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case UINT1:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new UInt1Accessor((UInt1Vector) vector);
        case OPTIONAL:
          return new NullableUInt1Accessor((NullableUInt1Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case UINT2:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new UInt2Accessor((UInt2Vector) vector);
        case OPTIONAL:
          return new NullableUInt2Accessor((NullableUInt2Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case SMALLINT:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new SmallIntAccessor((SmallIntVector) vector);
        case OPTIONAL:
          return new NullableSmallIntAccessor((NullableSmallIntVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case INT:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new IntAccessor((IntVector) vector);
        case OPTIONAL:
          return new NullableIntAccessor((NullableIntVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case UINT4:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new UInt4Accessor((UInt4Vector) vector);
        case OPTIONAL:
          return new NullableUInt4Accessor((NullableUInt4Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case FLOAT4:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Float4Accessor((Float4Vector) vector);
        case OPTIONAL:
          return new NullableFloat4Accessor((NullableFloat4Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case TIME:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new TimeAccessor((TimeVector) vector);
        case OPTIONAL:
          return new NullableTimeAccessor((NullableTimeVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case INTERVALYEAR:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new IntervalYearAccessor((IntervalYearVector) vector);
        case OPTIONAL:
          return new NullableIntervalYearAccessor((NullableIntervalYearVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL9:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal9Accessor((Decimal9Vector) vector);
        case OPTIONAL:
          return new NullableDecimal9Accessor((NullableDecimal9Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case BIGINT:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new BigIntAccessor((BigIntVector) vector);
        case OPTIONAL:
          return new NullableBigIntAccessor((NullableBigIntVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case UINT8:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new UInt8Accessor((UInt8Vector) vector);
        case OPTIONAL:
          return new NullableUInt8Accessor((NullableUInt8Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case FLOAT8:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Float8Accessor((Float8Vector) vector);
        case OPTIONAL:
          return new NullableFloat8Accessor((NullableFloat8Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DATE:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new DateAccessor((DateVector) vector);
        case OPTIONAL:
          return new NullableDateAccessor((NullableDateVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case TIMESTAMP:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new TimeStampAccessor((TimeStampVector) vector);
        case OPTIONAL:
          return new NullableTimeStampAccessor((NullableTimeStampVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL18:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal18Accessor((Decimal18Vector) vector);
        case OPTIONAL:
          return new NullableDecimal18Accessor((NullableDecimal18Vector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case TIMESTAMPTZ:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new TimeStampTZAccessor((TimeStampTZVector) vector);
        case OPTIONAL:
          return new NullableTimeStampTZAccessor((NullableTimeStampTZVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case INTERVALDAY:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new IntervalDayAccessor((IntervalDayVector) vector);
        case OPTIONAL:
          return new NullableIntervalDayAccessor((NullableIntervalDayVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case INTERVAL:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new IntervalAccessor((IntervalVector) vector);
        case OPTIONAL:
          return new NullableIntervalAccessor((NullableIntervalVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL28DENSE:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal28DenseAccessor((Decimal28DenseVector) vector);
        case OPTIONAL:
          return new NullableDecimal28DenseAccessor((NullableDecimal28DenseVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL38DENSE:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal38DenseAccessor((Decimal38DenseVector) vector);
        case OPTIONAL:
          return new NullableDecimal38DenseAccessor((NullableDecimal38DenseVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL38SPARSE:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal38SparseAccessor((Decimal38SparseVector) vector);
        case OPTIONAL:
          return new NullableDecimal38SparseAccessor((NullableDecimal38SparseVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case DECIMAL28SPARSE:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Decimal28SparseAccessor((Decimal28SparseVector) vector);
        case OPTIONAL:
          return new NullableDecimal28SparseAccessor((NullableDecimal28SparseVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case VARBINARY:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new VarBinaryAccessor((VarBinaryVector) vector);
        case OPTIONAL:
          return new NullableVarBinaryAccessor((NullableVarBinaryVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case VARCHAR:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new VarCharAccessor((VarCharVector) vector);
        case OPTIONAL:
          return new NullableVarCharAccessor((NullableVarCharVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case VAR16CHAR:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new Var16CharAccessor((Var16CharVector) vector);
        case OPTIONAL:
          return new NullableVar16CharAccessor((NullableVar16CharVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case BIT:
      switch (vector.getField().getType().getMode()) {
        case REQUIRED:
          return new BitAccessor((BitVector) vector);
        case OPTIONAL:
          return new NullableBitAccessor((NullableBitVector) vector);
        case REPEATED:
          return new GenericAccessor(vector);
      }
    case MAP:
    case LIST:
      return new GenericAccessor(vector);
    }

    throw new UnsupportedOperationException();
  }
  
  public static ValueVector getNewVector(SchemaPath parentPath, String name, BufferAllocator allocator, MajorType type){
    SchemaPath child = parentPath.getChild(name);
    MaterializedField field = MaterializedField.create(child, type);
    return getNewVector(field, allocator);
  }
  
  
  public static Class<?> getValueVectorClass(MinorType type, DataMode mode){
    switch (type) {
    case MAP:
      switch (mode) {
      case REQUIRED:
        return MapVector.class;
      case REPEATED:
        return RepeatedMapVector.class;
      }
      
    case LIST:
      switch (mode) {
      case REPEATED:
        return RepeatedListVector.class;
      }
    
      case TINYINT:
        switch (mode) {
          case REQUIRED:
            return TinyIntVector.class;
          case OPTIONAL:
            return NullableTinyIntVector.class;
          case REPEATED:
            return RepeatedTinyIntVector.class;
        }
      case UINT1:
        switch (mode) {
          case REQUIRED:
            return UInt1Vector.class;
          case OPTIONAL:
            return NullableUInt1Vector.class;
          case REPEATED:
            return RepeatedUInt1Vector.class;
        }
      case UINT2:
        switch (mode) {
          case REQUIRED:
            return UInt2Vector.class;
          case OPTIONAL:
            return NullableUInt2Vector.class;
          case REPEATED:
            return RepeatedUInt2Vector.class;
        }
      case SMALLINT:
        switch (mode) {
          case REQUIRED:
            return SmallIntVector.class;
          case OPTIONAL:
            return NullableSmallIntVector.class;
          case REPEATED:
            return RepeatedSmallIntVector.class;
        }
      case INT:
        switch (mode) {
          case REQUIRED:
            return IntVector.class;
          case OPTIONAL:
            return NullableIntVector.class;
          case REPEATED:
            return RepeatedIntVector.class;
        }
      case UINT4:
        switch (mode) {
          case REQUIRED:
            return UInt4Vector.class;
          case OPTIONAL:
            return NullableUInt4Vector.class;
          case REPEATED:
            return RepeatedUInt4Vector.class;
        }
      case FLOAT4:
        switch (mode) {
          case REQUIRED:
            return Float4Vector.class;
          case OPTIONAL:
            return NullableFloat4Vector.class;
          case REPEATED:
            return RepeatedFloat4Vector.class;
        }
      case TIME:
        switch (mode) {
          case REQUIRED:
            return TimeVector.class;
          case OPTIONAL:
            return NullableTimeVector.class;
          case REPEATED:
            return RepeatedTimeVector.class;
        }
      case INTERVALYEAR:
        switch (mode) {
          case REQUIRED:
            return IntervalYearVector.class;
          case OPTIONAL:
            return NullableIntervalYearVector.class;
          case REPEATED:
            return RepeatedIntervalYearVector.class;
        }
      case DECIMAL9:
        switch (mode) {
          case REQUIRED:
            return Decimal9Vector.class;
          case OPTIONAL:
            return NullableDecimal9Vector.class;
          case REPEATED:
            return RepeatedDecimal9Vector.class;
        }
      case BIGINT:
        switch (mode) {
          case REQUIRED:
            return BigIntVector.class;
          case OPTIONAL:
            return NullableBigIntVector.class;
          case REPEATED:
            return RepeatedBigIntVector.class;
        }
      case UINT8:
        switch (mode) {
          case REQUIRED:
            return UInt8Vector.class;
          case OPTIONAL:
            return NullableUInt8Vector.class;
          case REPEATED:
            return RepeatedUInt8Vector.class;
        }
      case FLOAT8:
        switch (mode) {
          case REQUIRED:
            return Float8Vector.class;
          case OPTIONAL:
            return NullableFloat8Vector.class;
          case REPEATED:
            return RepeatedFloat8Vector.class;
        }
      case DATE:
        switch (mode) {
          case REQUIRED:
            return DateVector.class;
          case OPTIONAL:
            return NullableDateVector.class;
          case REPEATED:
            return RepeatedDateVector.class;
        }
      case TIMESTAMP:
        switch (mode) {
          case REQUIRED:
            return TimeStampVector.class;
          case OPTIONAL:
            return NullableTimeStampVector.class;
          case REPEATED:
            return RepeatedTimeStampVector.class;
        }
      case DECIMAL18:
        switch (mode) {
          case REQUIRED:
            return Decimal18Vector.class;
          case OPTIONAL:
            return NullableDecimal18Vector.class;
          case REPEATED:
            return RepeatedDecimal18Vector.class;
        }
      case TIMESTAMPTZ:
        switch (mode) {
          case REQUIRED:
            return TimeStampTZVector.class;
          case OPTIONAL:
            return NullableTimeStampTZVector.class;
          case REPEATED:
            return RepeatedTimeStampTZVector.class;
        }
      case INTERVALDAY:
        switch (mode) {
          case REQUIRED:
            return IntervalDayVector.class;
          case OPTIONAL:
            return NullableIntervalDayVector.class;
          case REPEATED:
            return RepeatedIntervalDayVector.class;
        }
      case INTERVAL:
        switch (mode) {
          case REQUIRED:
            return IntervalVector.class;
          case OPTIONAL:
            return NullableIntervalVector.class;
          case REPEATED:
            return RepeatedIntervalVector.class;
        }
      case DECIMAL28DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28DenseVector.class;
          case OPTIONAL:
            return NullableDecimal28DenseVector.class;
          case REPEATED:
            return RepeatedDecimal28DenseVector.class;
        }
      case DECIMAL38DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38DenseVector.class;
          case OPTIONAL:
            return NullableDecimal38DenseVector.class;
          case REPEATED:
            return RepeatedDecimal38DenseVector.class;
        }
      case DECIMAL38SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38SparseVector.class;
          case OPTIONAL:
            return NullableDecimal38SparseVector.class;
          case REPEATED:
            return RepeatedDecimal38SparseVector.class;
        }
      case DECIMAL28SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28SparseVector.class;
          case OPTIONAL:
            return NullableDecimal28SparseVector.class;
          case REPEATED:
            return RepeatedDecimal28SparseVector.class;
        }
      case VARBINARY:
        switch (mode) {
          case REQUIRED:
            return VarBinaryVector.class;
          case OPTIONAL:
            return NullableVarBinaryVector.class;
          case REPEATED:
            return RepeatedVarBinaryVector.class;
        }
      case VARCHAR:
        switch (mode) {
          case REQUIRED:
            return VarCharVector.class;
          case OPTIONAL:
            return NullableVarCharVector.class;
          case REPEATED:
            return RepeatedVarCharVector.class;
        }
      case VAR16CHAR:
        switch (mode) {
          case REQUIRED:
            return Var16CharVector.class;
          case OPTIONAL:
            return NullableVar16CharVector.class;
          case REPEATED:
            return RepeatedVar16CharVector.class;
        }
      case BIT:
        switch (mode) {
          case REQUIRED:
            return BitVector.class;
          case OPTIONAL:
            return NullableBitVector.class;
          case REPEATED:
            return RepeatedBitVector.class;
        }
    case GENERIC_OBJECT      :
      return ObjectVector.class  ;
    default:
      break;
    }
    throw new UnsupportedOperationException();
  }
  public static Class<?> getReaderClassName( MinorType type, DataMode mode, boolean isSingularRepeated){
    switch (type) {
    case MAP:
      switch (mode) {
      case REQUIRED:
        if (!isSingularRepeated)
          return SingleMapReaderImpl.class;
        else
          return SingleLikeRepeatedMapReaderImpl.class;
      case REPEATED: 
          return RepeatedMapReaderImpl.class;
      }
    case LIST:
      switch (mode) {
      case REQUIRED:
        return SingleListReaderImpl.class;
      case REPEATED:
        return RepeatedListReaderImpl.class;
      }
      
      case TINYINT:
        switch (mode) {
          case REQUIRED:
            return TinyIntReaderImpl.class;
          case OPTIONAL:
            return NullableTinyIntReaderImpl.class;
          case REPEATED:
            return RepeatedTinyIntReaderImpl.class;
        }
      case UINT1:
        switch (mode) {
          case REQUIRED:
            return UInt1ReaderImpl.class;
          case OPTIONAL:
            return NullableUInt1ReaderImpl.class;
          case REPEATED:
            return RepeatedUInt1ReaderImpl.class;
        }
      case UINT2:
        switch (mode) {
          case REQUIRED:
            return UInt2ReaderImpl.class;
          case OPTIONAL:
            return NullableUInt2ReaderImpl.class;
          case REPEATED:
            return RepeatedUInt2ReaderImpl.class;
        }
      case SMALLINT:
        switch (mode) {
          case REQUIRED:
            return SmallIntReaderImpl.class;
          case OPTIONAL:
            return NullableSmallIntReaderImpl.class;
          case REPEATED:
            return RepeatedSmallIntReaderImpl.class;
        }
      case INT:
        switch (mode) {
          case REQUIRED:
            return IntReaderImpl.class;
          case OPTIONAL:
            return NullableIntReaderImpl.class;
          case REPEATED:
            return RepeatedIntReaderImpl.class;
        }
      case UINT4:
        switch (mode) {
          case REQUIRED:
            return UInt4ReaderImpl.class;
          case OPTIONAL:
            return NullableUInt4ReaderImpl.class;
          case REPEATED:
            return RepeatedUInt4ReaderImpl.class;
        }
      case FLOAT4:
        switch (mode) {
          case REQUIRED:
            return Float4ReaderImpl.class;
          case OPTIONAL:
            return NullableFloat4ReaderImpl.class;
          case REPEATED:
            return RepeatedFloat4ReaderImpl.class;
        }
      case TIME:
        switch (mode) {
          case REQUIRED:
            return TimeReaderImpl.class;
          case OPTIONAL:
            return NullableTimeReaderImpl.class;
          case REPEATED:
            return RepeatedTimeReaderImpl.class;
        }
      case INTERVALYEAR:
        switch (mode) {
          case REQUIRED:
            return IntervalYearReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalYearReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalYearReaderImpl.class;
        }
      case DECIMAL9:
        switch (mode) {
          case REQUIRED:
            return Decimal9ReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal9ReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal9ReaderImpl.class;
        }
      case BIGINT:
        switch (mode) {
          case REQUIRED:
            return BigIntReaderImpl.class;
          case OPTIONAL:
            return NullableBigIntReaderImpl.class;
          case REPEATED:
            return RepeatedBigIntReaderImpl.class;
        }
      case UINT8:
        switch (mode) {
          case REQUIRED:
            return UInt8ReaderImpl.class;
          case OPTIONAL:
            return NullableUInt8ReaderImpl.class;
          case REPEATED:
            return RepeatedUInt8ReaderImpl.class;
        }
      case FLOAT8:
        switch (mode) {
          case REQUIRED:
            return Float8ReaderImpl.class;
          case OPTIONAL:
            return NullableFloat8ReaderImpl.class;
          case REPEATED:
            return RepeatedFloat8ReaderImpl.class;
        }
      case DATE:
        switch (mode) {
          case REQUIRED:
            return DateReaderImpl.class;
          case OPTIONAL:
            return NullableDateReaderImpl.class;
          case REPEATED:
            return RepeatedDateReaderImpl.class;
        }
      case TIMESTAMP:
        switch (mode) {
          case REQUIRED:
            return TimeStampReaderImpl.class;
          case OPTIONAL:
            return NullableTimeStampReaderImpl.class;
          case REPEATED:
            return RepeatedTimeStampReaderImpl.class;
        }
      case DECIMAL18:
        switch (mode) {
          case REQUIRED:
            return Decimal18ReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal18ReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal18ReaderImpl.class;
        }
      case TIMESTAMPTZ:
        switch (mode) {
          case REQUIRED:
            return TimeStampTZReaderImpl.class;
          case OPTIONAL:
            return NullableTimeStampTZReaderImpl.class;
          case REPEATED:
            return RepeatedTimeStampTZReaderImpl.class;
        }
      case INTERVALDAY:
        switch (mode) {
          case REQUIRED:
            return IntervalDayReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalDayReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalDayReaderImpl.class;
        }
      case INTERVAL:
        switch (mode) {
          case REQUIRED:
            return IntervalReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalReaderImpl.class;
        }
      case DECIMAL28DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28DenseReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal28DenseReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal28DenseReaderImpl.class;
        }
      case DECIMAL38DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38DenseReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal38DenseReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal38DenseReaderImpl.class;
        }
      case DECIMAL38SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38SparseReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal38SparseReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal38SparseReaderImpl.class;
        }
      case DECIMAL28SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28SparseReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal28SparseReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal28SparseReaderImpl.class;
        }
      case VARBINARY:
        switch (mode) {
          case REQUIRED:
            return VarBinaryReaderImpl.class;
          case OPTIONAL:
            return NullableVarBinaryReaderImpl.class;
          case REPEATED:
            return RepeatedVarBinaryReaderImpl.class;
        }
      case VARCHAR:
        switch (mode) {
          case REQUIRED:
            return VarCharReaderImpl.class;
          case OPTIONAL:
            return NullableVarCharReaderImpl.class;
          case REPEATED:
            return RepeatedVarCharReaderImpl.class;
        }
      case VAR16CHAR:
        switch (mode) {
          case REQUIRED:
            return Var16CharReaderImpl.class;
          case OPTIONAL:
            return NullableVar16CharReaderImpl.class;
          case REPEATED:
            return RepeatedVar16CharReaderImpl.class;
        }
      case BIT:
        switch (mode) {
          case REQUIRED:
            return BitReaderImpl.class;
          case OPTIONAL:
            return NullableBitReaderImpl.class;
          case REPEATED:
            return RepeatedBitReaderImpl.class;
        }
      default:
        break;
      }
      throw new UnsupportedOperationException();    
  }
  
  public static Class<?> getWriterInterface( MinorType type, DataMode mode){
    switch (type) {
    case MAP: return MapWriter.class;
    case LIST: return ListWriter.class;
      case TINYINT: return TinyIntWriter.class;
      case UINT1: return UInt1Writer.class;
      case UINT2: return UInt2Writer.class;
      case SMALLINT: return SmallIntWriter.class;
      case INT: return IntWriter.class;
      case UINT4: return UInt4Writer.class;
      case FLOAT4: return Float4Writer.class;
      case TIME: return TimeWriter.class;
      case INTERVALYEAR: return IntervalYearWriter.class;
      case DECIMAL9: return Decimal9Writer.class;
      case BIGINT: return BigIntWriter.class;
      case UINT8: return UInt8Writer.class;
      case FLOAT8: return Float8Writer.class;
      case DATE: return DateWriter.class;
      case TIMESTAMP: return TimeStampWriter.class;
      case DECIMAL18: return Decimal18Writer.class;
      case TIMESTAMPTZ: return TimeStampTZWriter.class;
      case INTERVALDAY: return IntervalDayWriter.class;
      case INTERVAL: return IntervalWriter.class;
      case DECIMAL28DENSE: return Decimal28DenseWriter.class;
      case DECIMAL38DENSE: return Decimal38DenseWriter.class;
      case DECIMAL38SPARSE: return Decimal38SparseWriter.class;
      case DECIMAL28SPARSE: return Decimal28SparseWriter.class;
      case VARBINARY: return VarBinaryWriter.class;
      case VARCHAR: return VarCharWriter.class;
      case VAR16CHAR: return Var16CharWriter.class;
      case BIT: return BitWriter.class;
      default:
        break;
      }
      throw new UnsupportedOperationException();    
  }
  
  public static Class<?> getWriterImpl( MinorType type, DataMode mode){
    switch (type) {
    case MAP:
      switch (mode) {
      case REQUIRED:
        return SingleMapWriter.class;
      case REPEATED:
        return RepeatedMapWriter.class;
      }
    case LIST:
      switch (mode) {
      case REQUIRED:
        return SingleListWriter.class;
      case REPEATED:
        return RepeatedListWriter.class;
      }
      
      case TINYINT:
        switch (mode) {
          case REQUIRED:
            return TinyIntWriterImpl.class;
          case OPTIONAL:
            return NullableTinyIntWriterImpl.class;
          case REPEATED:
            return RepeatedTinyIntWriterImpl.class;
        }
      case UINT1:
        switch (mode) {
          case REQUIRED:
            return UInt1WriterImpl.class;
          case OPTIONAL:
            return NullableUInt1WriterImpl.class;
          case REPEATED:
            return RepeatedUInt1WriterImpl.class;
        }
      case UINT2:
        switch (mode) {
          case REQUIRED:
            return UInt2WriterImpl.class;
          case OPTIONAL:
            return NullableUInt2WriterImpl.class;
          case REPEATED:
            return RepeatedUInt2WriterImpl.class;
        }
      case SMALLINT:
        switch (mode) {
          case REQUIRED:
            return SmallIntWriterImpl.class;
          case OPTIONAL:
            return NullableSmallIntWriterImpl.class;
          case REPEATED:
            return RepeatedSmallIntWriterImpl.class;
        }
      case INT:
        switch (mode) {
          case REQUIRED:
            return IntWriterImpl.class;
          case OPTIONAL:
            return NullableIntWriterImpl.class;
          case REPEATED:
            return RepeatedIntWriterImpl.class;
        }
      case UINT4:
        switch (mode) {
          case REQUIRED:
            return UInt4WriterImpl.class;
          case OPTIONAL:
            return NullableUInt4WriterImpl.class;
          case REPEATED:
            return RepeatedUInt4WriterImpl.class;
        }
      case FLOAT4:
        switch (mode) {
          case REQUIRED:
            return Float4WriterImpl.class;
          case OPTIONAL:
            return NullableFloat4WriterImpl.class;
          case REPEATED:
            return RepeatedFloat4WriterImpl.class;
        }
      case TIME:
        switch (mode) {
          case REQUIRED:
            return TimeWriterImpl.class;
          case OPTIONAL:
            return NullableTimeWriterImpl.class;
          case REPEATED:
            return RepeatedTimeWriterImpl.class;
        }
      case INTERVALYEAR:
        switch (mode) {
          case REQUIRED:
            return IntervalYearWriterImpl.class;
          case OPTIONAL:
            return NullableIntervalYearWriterImpl.class;
          case REPEATED:
            return RepeatedIntervalYearWriterImpl.class;
        }
      case DECIMAL9:
        switch (mode) {
          case REQUIRED:
            return Decimal9WriterImpl.class;
          case OPTIONAL:
            return NullableDecimal9WriterImpl.class;
          case REPEATED:
            return RepeatedDecimal9WriterImpl.class;
        }
      case BIGINT:
        switch (mode) {
          case REQUIRED:
            return BigIntWriterImpl.class;
          case OPTIONAL:
            return NullableBigIntWriterImpl.class;
          case REPEATED:
            return RepeatedBigIntWriterImpl.class;
        }
      case UINT8:
        switch (mode) {
          case REQUIRED:
            return UInt8WriterImpl.class;
          case OPTIONAL:
            return NullableUInt8WriterImpl.class;
          case REPEATED:
            return RepeatedUInt8WriterImpl.class;
        }
      case FLOAT8:
        switch (mode) {
          case REQUIRED:
            return Float8WriterImpl.class;
          case OPTIONAL:
            return NullableFloat8WriterImpl.class;
          case REPEATED:
            return RepeatedFloat8WriterImpl.class;
        }
      case DATE:
        switch (mode) {
          case REQUIRED:
            return DateWriterImpl.class;
          case OPTIONAL:
            return NullableDateWriterImpl.class;
          case REPEATED:
            return RepeatedDateWriterImpl.class;
        }
      case TIMESTAMP:
        switch (mode) {
          case REQUIRED:
            return TimeStampWriterImpl.class;
          case OPTIONAL:
            return NullableTimeStampWriterImpl.class;
          case REPEATED:
            return RepeatedTimeStampWriterImpl.class;
        }
      case DECIMAL18:
        switch (mode) {
          case REQUIRED:
            return Decimal18WriterImpl.class;
          case OPTIONAL:
            return NullableDecimal18WriterImpl.class;
          case REPEATED:
            return RepeatedDecimal18WriterImpl.class;
        }
      case TIMESTAMPTZ:
        switch (mode) {
          case REQUIRED:
            return TimeStampTZWriterImpl.class;
          case OPTIONAL:
            return NullableTimeStampTZWriterImpl.class;
          case REPEATED:
            return RepeatedTimeStampTZWriterImpl.class;
        }
      case INTERVALDAY:
        switch (mode) {
          case REQUIRED:
            return IntervalDayWriterImpl.class;
          case OPTIONAL:
            return NullableIntervalDayWriterImpl.class;
          case REPEATED:
            return RepeatedIntervalDayWriterImpl.class;
        }
      case INTERVAL:
        switch (mode) {
          case REQUIRED:
            return IntervalWriterImpl.class;
          case OPTIONAL:
            return NullableIntervalWriterImpl.class;
          case REPEATED:
            return RepeatedIntervalWriterImpl.class;
        }
      case DECIMAL28DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28DenseWriterImpl.class;
          case OPTIONAL:
            return NullableDecimal28DenseWriterImpl.class;
          case REPEATED:
            return RepeatedDecimal28DenseWriterImpl.class;
        }
      case DECIMAL38DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38DenseWriterImpl.class;
          case OPTIONAL:
            return NullableDecimal38DenseWriterImpl.class;
          case REPEATED:
            return RepeatedDecimal38DenseWriterImpl.class;
        }
      case DECIMAL38SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38SparseWriterImpl.class;
          case OPTIONAL:
            return NullableDecimal38SparseWriterImpl.class;
          case REPEATED:
            return RepeatedDecimal38SparseWriterImpl.class;
        }
      case DECIMAL28SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28SparseWriterImpl.class;
          case OPTIONAL:
            return NullableDecimal28SparseWriterImpl.class;
          case REPEATED:
            return RepeatedDecimal28SparseWriterImpl.class;
        }
      case VARBINARY:
        switch (mode) {
          case REQUIRED:
            return VarBinaryWriterImpl.class;
          case OPTIONAL:
            return NullableVarBinaryWriterImpl.class;
          case REPEATED:
            return RepeatedVarBinaryWriterImpl.class;
        }
      case VARCHAR:
        switch (mode) {
          case REQUIRED:
            return VarCharWriterImpl.class;
          case OPTIONAL:
            return NullableVarCharWriterImpl.class;
          case REPEATED:
            return RepeatedVarCharWriterImpl.class;
        }
      case VAR16CHAR:
        switch (mode) {
          case REQUIRED:
            return Var16CharWriterImpl.class;
          case OPTIONAL:
            return NullableVar16CharWriterImpl.class;
          case REPEATED:
            return RepeatedVar16CharWriterImpl.class;
        }
      case BIT:
        switch (mode) {
          case REQUIRED:
            return BitWriterImpl.class;
          case OPTIONAL:
            return NullableBitWriterImpl.class;
          case REPEATED:
            return RepeatedBitWriterImpl.class;
        }
      default:
        break;
      }
      throw new UnsupportedOperationException();    
  }

  public static Class<?> getHolderReaderImpl( MinorType type, DataMode mode){
    switch (type) {      
      case TINYINT:
        switch (mode) {
          case REQUIRED:
            return TinyIntHolderReaderImpl.class;
          case OPTIONAL:
            return NullableTinyIntHolderReaderImpl.class;
          case REPEATED:
            return RepeatedTinyIntHolderReaderImpl.class;
        }
      case UINT1:
        switch (mode) {
          case REQUIRED:
            return UInt1HolderReaderImpl.class;
          case OPTIONAL:
            return NullableUInt1HolderReaderImpl.class;
          case REPEATED:
            return RepeatedUInt1HolderReaderImpl.class;
        }
      case UINT2:
        switch (mode) {
          case REQUIRED:
            return UInt2HolderReaderImpl.class;
          case OPTIONAL:
            return NullableUInt2HolderReaderImpl.class;
          case REPEATED:
            return RepeatedUInt2HolderReaderImpl.class;
        }
      case SMALLINT:
        switch (mode) {
          case REQUIRED:
            return SmallIntHolderReaderImpl.class;
          case OPTIONAL:
            return NullableSmallIntHolderReaderImpl.class;
          case REPEATED:
            return RepeatedSmallIntHolderReaderImpl.class;
        }
      case INT:
        switch (mode) {
          case REQUIRED:
            return IntHolderReaderImpl.class;
          case OPTIONAL:
            return NullableIntHolderReaderImpl.class;
          case REPEATED:
            return RepeatedIntHolderReaderImpl.class;
        }
      case UINT4:
        switch (mode) {
          case REQUIRED:
            return UInt4HolderReaderImpl.class;
          case OPTIONAL:
            return NullableUInt4HolderReaderImpl.class;
          case REPEATED:
            return RepeatedUInt4HolderReaderImpl.class;
        }
      case FLOAT4:
        switch (mode) {
          case REQUIRED:
            return Float4HolderReaderImpl.class;
          case OPTIONAL:
            return NullableFloat4HolderReaderImpl.class;
          case REPEATED:
            return RepeatedFloat4HolderReaderImpl.class;
        }
      case TIME:
        switch (mode) {
          case REQUIRED:
            return TimeHolderReaderImpl.class;
          case OPTIONAL:
            return NullableTimeHolderReaderImpl.class;
          case REPEATED:
            return RepeatedTimeHolderReaderImpl.class;
        }
      case INTERVALYEAR:
        switch (mode) {
          case REQUIRED:
            return IntervalYearHolderReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalYearHolderReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalYearHolderReaderImpl.class;
        }
      case DECIMAL9:
        switch (mode) {
          case REQUIRED:
            return Decimal9HolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal9HolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal9HolderReaderImpl.class;
        }
      case BIGINT:
        switch (mode) {
          case REQUIRED:
            return BigIntHolderReaderImpl.class;
          case OPTIONAL:
            return NullableBigIntHolderReaderImpl.class;
          case REPEATED:
            return RepeatedBigIntHolderReaderImpl.class;
        }
      case UINT8:
        switch (mode) {
          case REQUIRED:
            return UInt8HolderReaderImpl.class;
          case OPTIONAL:
            return NullableUInt8HolderReaderImpl.class;
          case REPEATED:
            return RepeatedUInt8HolderReaderImpl.class;
        }
      case FLOAT8:
        switch (mode) {
          case REQUIRED:
            return Float8HolderReaderImpl.class;
          case OPTIONAL:
            return NullableFloat8HolderReaderImpl.class;
          case REPEATED:
            return RepeatedFloat8HolderReaderImpl.class;
        }
      case DATE:
        switch (mode) {
          case REQUIRED:
            return DateHolderReaderImpl.class;
          case OPTIONAL:
            return NullableDateHolderReaderImpl.class;
          case REPEATED:
            return RepeatedDateHolderReaderImpl.class;
        }
      case TIMESTAMP:
        switch (mode) {
          case REQUIRED:
            return TimeStampHolderReaderImpl.class;
          case OPTIONAL:
            return NullableTimeStampHolderReaderImpl.class;
          case REPEATED:
            return RepeatedTimeStampHolderReaderImpl.class;
        }
      case DECIMAL18:
        switch (mode) {
          case REQUIRED:
            return Decimal18HolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal18HolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal18HolderReaderImpl.class;
        }
      case TIMESTAMPTZ:
        switch (mode) {
          case REQUIRED:
            return TimeStampTZHolderReaderImpl.class;
          case OPTIONAL:
            return NullableTimeStampTZHolderReaderImpl.class;
          case REPEATED:
            return RepeatedTimeStampTZHolderReaderImpl.class;
        }
      case INTERVALDAY:
        switch (mode) {
          case REQUIRED:
            return IntervalDayHolderReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalDayHolderReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalDayHolderReaderImpl.class;
        }
      case INTERVAL:
        switch (mode) {
          case REQUIRED:
            return IntervalHolderReaderImpl.class;
          case OPTIONAL:
            return NullableIntervalHolderReaderImpl.class;
          case REPEATED:
            return RepeatedIntervalHolderReaderImpl.class;
        }
      case DECIMAL28DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28DenseHolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal28DenseHolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal28DenseHolderReaderImpl.class;
        }
      case DECIMAL38DENSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38DenseHolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal38DenseHolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal38DenseHolderReaderImpl.class;
        }
      case DECIMAL38SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal38SparseHolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal38SparseHolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal38SparseHolderReaderImpl.class;
        }
      case DECIMAL28SPARSE:
        switch (mode) {
          case REQUIRED:
            return Decimal28SparseHolderReaderImpl.class;
          case OPTIONAL:
            return NullableDecimal28SparseHolderReaderImpl.class;
          case REPEATED:
            return RepeatedDecimal28SparseHolderReaderImpl.class;
        }
      case VARBINARY:
        switch (mode) {
          case REQUIRED:
            return VarBinaryHolderReaderImpl.class;
          case OPTIONAL:
            return NullableVarBinaryHolderReaderImpl.class;
          case REPEATED:
            return RepeatedVarBinaryHolderReaderImpl.class;
        }
      case VARCHAR:
        switch (mode) {
          case REQUIRED:
            return VarCharHolderReaderImpl.class;
          case OPTIONAL:
            return NullableVarCharHolderReaderImpl.class;
          case REPEATED:
            return RepeatedVarCharHolderReaderImpl.class;
        }
      case VAR16CHAR:
        switch (mode) {
          case REQUIRED:
            return Var16CharHolderReaderImpl.class;
          case OPTIONAL:
            return NullableVar16CharHolderReaderImpl.class;
          case REPEATED:
            return RepeatedVar16CharHolderReaderImpl.class;
        }
      case BIT:
        switch (mode) {
          case REQUIRED:
            return BitHolderReaderImpl.class;
          case OPTIONAL:
            return NullableBitHolderReaderImpl.class;
          case REPEATED:
            return RepeatedBitHolderReaderImpl.class;
        }
      default:
        break;
      }
      throw new UnsupportedOperationException();    
  }
  
  public static JType getHolderType(JCodeModel model, MinorType type, DataMode mode){
    switch (type) {
    case MAP:
    case LIST:
      return model._ref(ComplexHolder.class);
      
      case TINYINT:
        switch (mode) {
          case REQUIRED:
            return model._ref(TinyIntHolder.class);
          case OPTIONAL:
            return model._ref(NullableTinyIntHolder.class);
          case REPEATED:
            return model._ref(RepeatedTinyIntHolder.class);
        }
      case UINT1:
        switch (mode) {
          case REQUIRED:
            return model._ref(UInt1Holder.class);
          case OPTIONAL:
            return model._ref(NullableUInt1Holder.class);
          case REPEATED:
            return model._ref(RepeatedUInt1Holder.class);
        }
      case UINT2:
        switch (mode) {
          case REQUIRED:
            return model._ref(UInt2Holder.class);
          case OPTIONAL:
            return model._ref(NullableUInt2Holder.class);
          case REPEATED:
            return model._ref(RepeatedUInt2Holder.class);
        }
      case SMALLINT:
        switch (mode) {
          case REQUIRED:
            return model._ref(SmallIntHolder.class);
          case OPTIONAL:
            return model._ref(NullableSmallIntHolder.class);
          case REPEATED:
            return model._ref(RepeatedSmallIntHolder.class);
        }
      case INT:
        switch (mode) {
          case REQUIRED:
            return model._ref(IntHolder.class);
          case OPTIONAL:
            return model._ref(NullableIntHolder.class);
          case REPEATED:
            return model._ref(RepeatedIntHolder.class);
        }
      case UINT4:
        switch (mode) {
          case REQUIRED:
            return model._ref(UInt4Holder.class);
          case OPTIONAL:
            return model._ref(NullableUInt4Holder.class);
          case REPEATED:
            return model._ref(RepeatedUInt4Holder.class);
        }
      case FLOAT4:
        switch (mode) {
          case REQUIRED:
            return model._ref(Float4Holder.class);
          case OPTIONAL:
            return model._ref(NullableFloat4Holder.class);
          case REPEATED:
            return model._ref(RepeatedFloat4Holder.class);
        }
      case TIME:
        switch (mode) {
          case REQUIRED:
            return model._ref(TimeHolder.class);
          case OPTIONAL:
            return model._ref(NullableTimeHolder.class);
          case REPEATED:
            return model._ref(RepeatedTimeHolder.class);
        }
      case INTERVALYEAR:
        switch (mode) {
          case REQUIRED:
            return model._ref(IntervalYearHolder.class);
          case OPTIONAL:
            return model._ref(NullableIntervalYearHolder.class);
          case REPEATED:
            return model._ref(RepeatedIntervalYearHolder.class);
        }
      case DECIMAL9:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal9Holder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal9Holder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal9Holder.class);
        }
      case BIGINT:
        switch (mode) {
          case REQUIRED:
            return model._ref(BigIntHolder.class);
          case OPTIONAL:
            return model._ref(NullableBigIntHolder.class);
          case REPEATED:
            return model._ref(RepeatedBigIntHolder.class);
        }
      case UINT8:
        switch (mode) {
          case REQUIRED:
            return model._ref(UInt8Holder.class);
          case OPTIONAL:
            return model._ref(NullableUInt8Holder.class);
          case REPEATED:
            return model._ref(RepeatedUInt8Holder.class);
        }
      case FLOAT8:
        switch (mode) {
          case REQUIRED:
            return model._ref(Float8Holder.class);
          case OPTIONAL:
            return model._ref(NullableFloat8Holder.class);
          case REPEATED:
            return model._ref(RepeatedFloat8Holder.class);
        }
      case DATE:
        switch (mode) {
          case REQUIRED:
            return model._ref(DateHolder.class);
          case OPTIONAL:
            return model._ref(NullableDateHolder.class);
          case REPEATED:
            return model._ref(RepeatedDateHolder.class);
        }
      case TIMESTAMP:
        switch (mode) {
          case REQUIRED:
            return model._ref(TimeStampHolder.class);
          case OPTIONAL:
            return model._ref(NullableTimeStampHolder.class);
          case REPEATED:
            return model._ref(RepeatedTimeStampHolder.class);
        }
      case DECIMAL18:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal18Holder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal18Holder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal18Holder.class);
        }
      case TIMESTAMPTZ:
        switch (mode) {
          case REQUIRED:
            return model._ref(TimeStampTZHolder.class);
          case OPTIONAL:
            return model._ref(NullableTimeStampTZHolder.class);
          case REPEATED:
            return model._ref(RepeatedTimeStampTZHolder.class);
        }
      case INTERVALDAY:
        switch (mode) {
          case REQUIRED:
            return model._ref(IntervalDayHolder.class);
          case OPTIONAL:
            return model._ref(NullableIntervalDayHolder.class);
          case REPEATED:
            return model._ref(RepeatedIntervalDayHolder.class);
        }
      case INTERVAL:
        switch (mode) {
          case REQUIRED:
            return model._ref(IntervalHolder.class);
          case OPTIONAL:
            return model._ref(NullableIntervalHolder.class);
          case REPEATED:
            return model._ref(RepeatedIntervalHolder.class);
        }
      case DECIMAL28DENSE:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal28DenseHolder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal28DenseHolder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal28DenseHolder.class);
        }
      case DECIMAL38DENSE:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal38DenseHolder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal38DenseHolder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal38DenseHolder.class);
        }
      case DECIMAL38SPARSE:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal38SparseHolder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal38SparseHolder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal38SparseHolder.class);
        }
      case DECIMAL28SPARSE:
        switch (mode) {
          case REQUIRED:
            return model._ref(Decimal28SparseHolder.class);
          case OPTIONAL:
            return model._ref(NullableDecimal28SparseHolder.class);
          case REPEATED:
            return model._ref(RepeatedDecimal28SparseHolder.class);
        }
      case VARBINARY:
        switch (mode) {
          case REQUIRED:
            return model._ref(VarBinaryHolder.class);
          case OPTIONAL:
            return model._ref(NullableVarBinaryHolder.class);
          case REPEATED:
            return model._ref(RepeatedVarBinaryHolder.class);
        }
      case VARCHAR:
        switch (mode) {
          case REQUIRED:
            return model._ref(VarCharHolder.class);
          case OPTIONAL:
            return model._ref(NullableVarCharHolder.class);
          case REPEATED:
            return model._ref(RepeatedVarCharHolder.class);
        }
      case VAR16CHAR:
        switch (mode) {
          case REQUIRED:
            return model._ref(Var16CharHolder.class);
          case OPTIONAL:
            return model._ref(NullableVar16CharHolder.class);
          case REPEATED:
            return model._ref(RepeatedVar16CharHolder.class);
        }
      case BIT:
        switch (mode) {
          case REQUIRED:
            return model._ref(BitHolder.class);
          case OPTIONAL:
            return model._ref(NullableBitHolder.class);
          case REPEATED:
            return model._ref(RepeatedBitHolder.class);
        }
      case GENERIC_OBJECT:
        return model._ref(ObjectHolder.class);
      default:
        break;
      }
      throw new UnsupportedOperationException();
  }

  public static ValueVector getNewVector(MaterializedField field, BufferAllocator allocator){
    MajorType type = field.getType();

    switch (type.getMinorType()) {
    
    
    case MAP:
      switch (type.getMode()) {
      case REQUIRED:
        return new MapVector(field, allocator);
      case REPEATED:
        return new RepeatedMapVector(field, allocator);
      }
    case LIST:
      switch (type.getMode()) {
      case REPEATED:
        return new RepeatedListVector(field, allocator);
      }    
    case TINYINT:
      switch (type.getMode()) {
        case REQUIRED:
          return new TinyIntVector(field, allocator);
        case OPTIONAL:
          return new NullableTinyIntVector(field, allocator);
        case REPEATED:
          return new RepeatedTinyIntVector(field, allocator);
      }
    case UINT1:
      switch (type.getMode()) {
        case REQUIRED:
          return new UInt1Vector(field, allocator);
        case OPTIONAL:
          return new NullableUInt1Vector(field, allocator);
        case REPEATED:
          return new RepeatedUInt1Vector(field, allocator);
      }
    case UINT2:
      switch (type.getMode()) {
        case REQUIRED:
          return new UInt2Vector(field, allocator);
        case OPTIONAL:
          return new NullableUInt2Vector(field, allocator);
        case REPEATED:
          return new RepeatedUInt2Vector(field, allocator);
      }
    case SMALLINT:
      switch (type.getMode()) {
        case REQUIRED:
          return new SmallIntVector(field, allocator);
        case OPTIONAL:
          return new NullableSmallIntVector(field, allocator);
        case REPEATED:
          return new RepeatedSmallIntVector(field, allocator);
      }
    case INT:
      switch (type.getMode()) {
        case REQUIRED:
          return new IntVector(field, allocator);
        case OPTIONAL:
          return new NullableIntVector(field, allocator);
        case REPEATED:
          return new RepeatedIntVector(field, allocator);
      }
    case UINT4:
      switch (type.getMode()) {
        case REQUIRED:
          return new UInt4Vector(field, allocator);
        case OPTIONAL:
          return new NullableUInt4Vector(field, allocator);
        case REPEATED:
          return new RepeatedUInt4Vector(field, allocator);
      }
    case FLOAT4:
      switch (type.getMode()) {
        case REQUIRED:
          return new Float4Vector(field, allocator);
        case OPTIONAL:
          return new NullableFloat4Vector(field, allocator);
        case REPEATED:
          return new RepeatedFloat4Vector(field, allocator);
      }
    case TIME:
      switch (type.getMode()) {
        case REQUIRED:
          return new TimeVector(field, allocator);
        case OPTIONAL:
          return new NullableTimeVector(field, allocator);
        case REPEATED:
          return new RepeatedTimeVector(field, allocator);
      }
    case INTERVALYEAR:
      switch (type.getMode()) {
        case REQUIRED:
          return new IntervalYearVector(field, allocator);
        case OPTIONAL:
          return new NullableIntervalYearVector(field, allocator);
        case REPEATED:
          return new RepeatedIntervalYearVector(field, allocator);
      }
    case DECIMAL9:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal9Vector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal9Vector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal9Vector(field, allocator);
      }
    case BIGINT:
      switch (type.getMode()) {
        case REQUIRED:
          return new BigIntVector(field, allocator);
        case OPTIONAL:
          return new NullableBigIntVector(field, allocator);
        case REPEATED:
          return new RepeatedBigIntVector(field, allocator);
      }
    case UINT8:
      switch (type.getMode()) {
        case REQUIRED:
          return new UInt8Vector(field, allocator);
        case OPTIONAL:
          return new NullableUInt8Vector(field, allocator);
        case REPEATED:
          return new RepeatedUInt8Vector(field, allocator);
      }
    case FLOAT8:
      switch (type.getMode()) {
        case REQUIRED:
          return new Float8Vector(field, allocator);
        case OPTIONAL:
          return new NullableFloat8Vector(field, allocator);
        case REPEATED:
          return new RepeatedFloat8Vector(field, allocator);
      }
    case DATE:
      switch (type.getMode()) {
        case REQUIRED:
          return new DateVector(field, allocator);
        case OPTIONAL:
          return new NullableDateVector(field, allocator);
        case REPEATED:
          return new RepeatedDateVector(field, allocator);
      }
    case TIMESTAMP:
      switch (type.getMode()) {
        case REQUIRED:
          return new TimeStampVector(field, allocator);
        case OPTIONAL:
          return new NullableTimeStampVector(field, allocator);
        case REPEATED:
          return new RepeatedTimeStampVector(field, allocator);
      }
    case DECIMAL18:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal18Vector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal18Vector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal18Vector(field, allocator);
      }
    case TIMESTAMPTZ:
      switch (type.getMode()) {
        case REQUIRED:
          return new TimeStampTZVector(field, allocator);
        case OPTIONAL:
          return new NullableTimeStampTZVector(field, allocator);
        case REPEATED:
          return new RepeatedTimeStampTZVector(field, allocator);
      }
    case INTERVALDAY:
      switch (type.getMode()) {
        case REQUIRED:
          return new IntervalDayVector(field, allocator);
        case OPTIONAL:
          return new NullableIntervalDayVector(field, allocator);
        case REPEATED:
          return new RepeatedIntervalDayVector(field, allocator);
      }
    case INTERVAL:
      switch (type.getMode()) {
        case REQUIRED:
          return new IntervalVector(field, allocator);
        case OPTIONAL:
          return new NullableIntervalVector(field, allocator);
        case REPEATED:
          return new RepeatedIntervalVector(field, allocator);
      }
    case DECIMAL28DENSE:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal28DenseVector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal28DenseVector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal28DenseVector(field, allocator);
      }
    case DECIMAL38DENSE:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal38DenseVector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal38DenseVector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal38DenseVector(field, allocator);
      }
    case DECIMAL38SPARSE:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal38SparseVector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal38SparseVector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal38SparseVector(field, allocator);
      }
    case DECIMAL28SPARSE:
      switch (type.getMode()) {
        case REQUIRED:
          return new Decimal28SparseVector(field, allocator);
        case OPTIONAL:
          return new NullableDecimal28SparseVector(field, allocator);
        case REPEATED:
          return new RepeatedDecimal28SparseVector(field, allocator);
      }
    case VARBINARY:
      switch (type.getMode()) {
        case REQUIRED:
          return new VarBinaryVector(field, allocator);
        case OPTIONAL:
          return new NullableVarBinaryVector(field, allocator);
        case REPEATED:
          return new RepeatedVarBinaryVector(field, allocator);
      }
    case VARCHAR:
      switch (type.getMode()) {
        case REQUIRED:
          return new VarCharVector(field, allocator);
        case OPTIONAL:
          return new NullableVarCharVector(field, allocator);
        case REPEATED:
          return new RepeatedVarCharVector(field, allocator);
      }
    case VAR16CHAR:
      switch (type.getMode()) {
        case REQUIRED:
          return new Var16CharVector(field, allocator);
        case OPTIONAL:
          return new NullableVar16CharVector(field, allocator);
        case REPEATED:
          return new RepeatedVar16CharVector(field, allocator);
      }
    case BIT:
      switch (type.getMode()) {
        case REQUIRED:
          return new BitVector(field, allocator);
        case OPTIONAL:
          return new NullableBitVector(field, allocator);
        case REPEATED:
          return new RepeatedBitVector(field, allocator);
      }
    case GENERIC_OBJECT:
      return new ObjectVector(field, allocator)        ;
    default:
      break;
    }
    // All ValueVector types have been handled.
    throw new UnsupportedOperationException(type.getMinorType() + " type is not supported. Mode: " + type.getMode());
  }

  public static ValueHolder getValue(ValueVector vector, int index) {
    MajorType type = vector.getField().getType();
    ValueHolder holder;
    switch(type.getMinorType()) {
    case TINYINT :
      holder = new TinyIntHolder(); 
      ((TinyIntHolder)holder).value = ((TinyIntVector) vector).getAccessor().get(index);
      break;
    case UINT1 :
      holder = new UInt1Holder(); 
      ((UInt1Holder)holder).value = ((UInt1Vector) vector).getAccessor().get(index);
      break;
    case UINT2 :
      holder = new UInt2Holder(); 
      ((UInt2Holder)holder).value = ((UInt2Vector) vector).getAccessor().get(index);
      break;
    case SMALLINT :
      holder = new SmallIntHolder(); 
      ((SmallIntHolder)holder).value = ((SmallIntVector) vector).getAccessor().get(index);
      break;
    case INT :
      holder = new IntHolder(); 
      ((IntHolder)holder).value = ((IntVector) vector).getAccessor().get(index);
      break;
    case UINT4 :
      holder = new UInt4Holder(); 
      ((UInt4Holder)holder).value = ((UInt4Vector) vector).getAccessor().get(index);
      break;
    case FLOAT4 :
      holder = new Float4Holder(); 
      ((Float4Holder)holder).value = ((Float4Vector) vector).getAccessor().get(index);
      break;
    case TIME :
      holder = new TimeHolder(); 
      ((TimeHolder)holder).value = ((TimeVector) vector).getAccessor().get(index);
      break;
    case INTERVALYEAR :
      holder = new IntervalYearHolder(); 
      ((IntervalYearHolder)holder).value = ((IntervalYearVector) vector).getAccessor().get(index);
      break;
    case DECIMAL9 :
      holder = new Decimal9Holder(); 
      ((Decimal9Holder)holder).value = ((Decimal9Vector) vector).getAccessor().get(index);
      break;
    case BIGINT :
      holder = new BigIntHolder(); 
      ((BigIntHolder)holder).value = ((BigIntVector) vector).getAccessor().get(index);
      break;
    case UINT8 :
      holder = new UInt8Holder(); 
      ((UInt8Holder)holder).value = ((UInt8Vector) vector).getAccessor().get(index);
      break;
    case FLOAT8 :
      holder = new Float8Holder(); 
      ((Float8Holder)holder).value = ((Float8Vector) vector).getAccessor().get(index);
      break;
    case DATE :
      holder = new DateHolder(); 
      ((DateHolder)holder).value = ((DateVector) vector).getAccessor().get(index);
      break;
    case TIMESTAMP :
      holder = new TimeStampHolder(); 
      ((TimeStampHolder)holder).value = ((TimeStampVector) vector).getAccessor().get(index);
      break;
    case DECIMAL18 :
      holder = new Decimal18Holder(); 
      ((Decimal18Holder)holder).value = ((Decimal18Vector) vector).getAccessor().get(index);
      break;
    case TIMESTAMPTZ :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case INTERVALDAY :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case INTERVAL :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case DECIMAL28DENSE :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case DECIMAL38DENSE :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case DECIMAL38SPARSE :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case DECIMAL28SPARSE :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case VARBINARY :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case VARCHAR :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case VAR16CHAR :
         throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");
    case BIT :
      holder = new BitHolder(); 
      ((BitHolder)holder).value = ((BitVector) vector).getAccessor().get(index);
      break;
    case GENERIC_OBJECT:
      holder = new ObjectHolder();
      ((ObjectHolder)holder).obj = ((ObjectVector) vector).getAccessor().getObject(index)         ;
      break;
    default:
      throw new UnsupportedOperationException(type.getMinorType() + " type is not supported."); 
    }
    return holder;
  }

  public static void setValue(ValueVector vector, int index, ValueHolder holder) {
    MajorType type = vector.getField().getType();

    switch(type.getMinorType()) {
    case TINYINT :
      ((TinyIntVector) vector).getMutator().setSafe(index, (TinyIntHolder) holder);
      break;
    case UINT1 :
      ((UInt1Vector) vector).getMutator().setSafe(index, (UInt1Holder) holder);
      break;
    case UINT2 :
      ((UInt2Vector) vector).getMutator().setSafe(index, (UInt2Holder) holder);
      break;
    case SMALLINT :
      ((SmallIntVector) vector).getMutator().setSafe(index, (SmallIntHolder) holder);
      break;
    case INT :
      ((IntVector) vector).getMutator().setSafe(index, (IntHolder) holder);
      break;
    case UINT4 :
      ((UInt4Vector) vector).getMutator().setSafe(index, (UInt4Holder) holder);
      break;
    case FLOAT4 :
      ((Float4Vector) vector).getMutator().setSafe(index, (Float4Holder) holder);
      break;
    case TIME :
      ((TimeVector) vector).getMutator().setSafe(index, (TimeHolder) holder);
      break;
    case INTERVALYEAR :
      ((IntervalYearVector) vector).getMutator().setSafe(index, (IntervalYearHolder) holder);
      break;
    case DECIMAL9 :
      ((Decimal9Vector) vector).getMutator().setSafe(index, (Decimal9Holder) holder);
      break;
    case BIGINT :
      ((BigIntVector) vector).getMutator().setSafe(index, (BigIntHolder) holder);
      break;
    case UINT8 :
      ((UInt8Vector) vector).getMutator().setSafe(index, (UInt8Holder) holder);
      break;
    case FLOAT8 :
      ((Float8Vector) vector).getMutator().setSafe(index, (Float8Holder) holder);
      break;
    case DATE :
      ((DateVector) vector).getMutator().setSafe(index, (DateHolder) holder);
      break;
    case TIMESTAMP :
      ((TimeStampVector) vector).getMutator().setSafe(index, (TimeStampHolder) holder);
      break;
    case DECIMAL18 :
      ((Decimal18Vector) vector).getMutator().setSafe(index, (Decimal18Holder) holder);
      break;
    case TIMESTAMPTZ :
      ((TimeStampTZVector) vector).getMutator().setSafe(index, (TimeStampTZHolder) holder);
      break;
    case INTERVALDAY :
      ((IntervalDayVector) vector).getMutator().setSafe(index, (IntervalDayHolder) holder);
      break;
    case INTERVAL :
      ((IntervalVector) vector).getMutator().setSafe(index, (IntervalHolder) holder);
      break;
    case DECIMAL28DENSE :
      ((Decimal28DenseVector) vector).getMutator().setSafe(index, (Decimal28DenseHolder) holder);
      break;
    case DECIMAL38DENSE :
      ((Decimal38DenseVector) vector).getMutator().setSafe(index, (Decimal38DenseHolder) holder);
      break;
    case DECIMAL38SPARSE :
      ((Decimal38SparseVector) vector).getMutator().setSafe(index, (Decimal38SparseHolder) holder);
      break;
    case DECIMAL28SPARSE :
      ((Decimal28SparseVector) vector).getMutator().setSafe(index, (Decimal28SparseHolder) holder);
      break;
    case VARBINARY :
      ((VarBinaryVector) vector).getMutator().setSafe(index, (VarBinaryHolder) holder);
      break;
    case VARCHAR :
      ((VarCharVector) vector).getMutator().setSafe(index, (VarCharHolder) holder);
      break;
    case VAR16CHAR :
      ((Var16CharVector) vector).getMutator().setSafe(index, (Var16CharHolder) holder);
      break;
    case BIT :
      ((BitVector) vector).getMutator().setSafe(index, (BitHolder) holder);
      break;
    case GENERIC_OBJECT:
      ((ObjectVector) vector).getMutator().setSafe(index, (ObjectHolder) holder);
      break          ;
    default:
      throw new UnsupportedOperationException(type.getMinorType() + " type is not supported.");    
    }
  }

  public static boolean compareValues(ValueVector v1, int v1index, ValueVector v2, int v2index) {
    MajorType type1 = v1.getField().getType();
    MajorType type2 = v2.getField().getType();

    if (type1.getMinorType() != type2.getMinorType()) {
      return false;
    }

    switch(type1.getMinorType()) {
    case TINYINT :
      if ( ((TinyIntVector) v1).getAccessor().get(v1index) == 
           ((TinyIntVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case UINT1 :
      if ( ((UInt1Vector) v1).getAccessor().get(v1index) == 
           ((UInt1Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case UINT2 :
      if ( ((UInt2Vector) v1).getAccessor().get(v1index) == 
           ((UInt2Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case SMALLINT :
      if ( ((SmallIntVector) v1).getAccessor().get(v1index) == 
           ((SmallIntVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case INT :
      if ( ((IntVector) v1).getAccessor().get(v1index) == 
           ((IntVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case UINT4 :
      if ( ((UInt4Vector) v1).getAccessor().get(v1index) == 
           ((UInt4Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case FLOAT4 :
      if ( ((Float4Vector) v1).getAccessor().get(v1index) == 
           ((Float4Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case TIME :
      if ( ((TimeVector) v1).getAccessor().get(v1index) == 
           ((TimeVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case INTERVALYEAR :
      if ( ((IntervalYearVector) v1).getAccessor().get(v1index) == 
           ((IntervalYearVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL9 :
      if ( ((Decimal9Vector) v1).getAccessor().get(v1index) == 
           ((Decimal9Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case BIGINT :
      if ( ((BigIntVector) v1).getAccessor().get(v1index) == 
           ((BigIntVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case UINT8 :
      if ( ((UInt8Vector) v1).getAccessor().get(v1index) == 
           ((UInt8Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case FLOAT8 :
      if ( ((Float8Vector) v1).getAccessor().get(v1index) == 
           ((Float8Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DATE :
      if ( ((DateVector) v1).getAccessor().get(v1index) == 
           ((DateVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case TIMESTAMP :
      if ( ((TimeStampVector) v1).getAccessor().get(v1index) == 
           ((TimeStampVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL18 :
      if ( ((Decimal18Vector) v1).getAccessor().get(v1index) == 
           ((Decimal18Vector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case TIMESTAMPTZ :
      if ( ((TimeStampTZVector) v1).getAccessor().get(v1index) == 
           ((TimeStampTZVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case INTERVALDAY :
      if ( ((IntervalDayVector) v1).getAccessor().get(v1index) == 
           ((IntervalDayVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case INTERVAL :
      if ( ((IntervalVector) v1).getAccessor().get(v1index) == 
           ((IntervalVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL28DENSE :
      if ( ((Decimal28DenseVector) v1).getAccessor().get(v1index) == 
           ((Decimal28DenseVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL38DENSE :
      if ( ((Decimal38DenseVector) v1).getAccessor().get(v1index) == 
           ((Decimal38DenseVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL38SPARSE :
      if ( ((Decimal38SparseVector) v1).getAccessor().get(v1index) == 
           ((Decimal38SparseVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case DECIMAL28SPARSE :
      if ( ((Decimal28SparseVector) v1).getAccessor().get(v1index) == 
           ((Decimal28SparseVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case VARBINARY :
      if ( ((VarBinaryVector) v1).getAccessor().get(v1index) == 
           ((VarBinaryVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case VARCHAR :
      if ( ((VarCharVector) v1).getAccessor().get(v1index) == 
           ((VarCharVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case VAR16CHAR :
      if ( ((Var16CharVector) v1).getAccessor().get(v1index) == 
           ((Var16CharVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    case BIT :
      if ( ((BitVector) v1).getAccessor().get(v1index) == 
           ((BitVector) v2).getAccessor().get(v2index) ) 
        return true;
      break;
    default:
      break;
    }
    return false;
  }

}
