/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.memory;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Maps;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.DrillBuf;
import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.ConcurrentMap;
import org.apache.drill.exec.memory.AtomicRemainder;
import org.apache.drill.exec.proto.ExecProtos;
import org.apache.drill.exec.proto.helper.QueryIdHelper;
import org.apache.drill.exec.util.AssertionUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Accountor {
    static final Logger logger = LoggerFactory.getLogger(Accountor.class);
    private static final boolean ENABLE_ACCOUNTING = AssertionUtil.isAssertionsEnabled();
    private final AtomicRemainder remainder;
    private final long total;
    private ConcurrentMap<ByteBuf, DebugStackTrace> buffers = Maps.newConcurrentMap();
    private final ExecProtos.FragmentHandle handle;
    private Accountor parent;
    private final boolean errorOnLeak;

    public Accountor(boolean errorOnLeak, ExecProtos.FragmentHandle handle, Accountor parent, long max, long preAllocated) {
        this.errorOnLeak = errorOnLeak;
        AtomicRemainder parentRemainder = parent != null ? parent.remainder : null;
        this.parent = parent;
        this.remainder = new AtomicRemainder(errorOnLeak, parentRemainder, max, preAllocated);
        this.total = max;
        this.handle = handle;
        this.buffers = ENABLE_ACCOUNTING ? Maps.newConcurrentMap() : null;
    }

    public boolean transferTo(Accountor target, DrillBuf buf, long size) {
        boolean withinLimit = target.forceAdditionalReservation(size);
        this.release(buf, size);
        if (ENABLE_ACCOUNTING) {
            target.buffers.put(buf, new DebugStackTrace(buf.capacity(), Thread.currentThread().getStackTrace()));
        }
        return withinLimit;
    }

    public long getAvailable() {
        if (this.parent != null) {
            return Math.min(this.parent.getAvailable(), this.getCapacity() - this.getAllocation());
        }
        return this.getCapacity() - this.getAllocation();
    }

    public long getCapacity() {
        return this.total;
    }

    public long getAllocation() {
        return this.remainder.getUsed();
    }

    public boolean reserve(long size) {
        return this.remainder.get(size);
    }

    public boolean forceAdditionalReservation(long size) {
        if (size > 0L) {
            return this.remainder.forceGet(size);
        }
        return true;
    }

    public void reserved(long expected, DrillBuf buf) {
        long additional = (long)buf.capacity() - expected;
        if (additional > 0L) {
            this.remainder.forceGet(additional);
        }
        if (ENABLE_ACCOUNTING) {
            this.buffers.put(buf, new DebugStackTrace(buf.capacity(), Thread.currentThread().getStackTrace()));
        }
    }

    public void releasePartial(DrillBuf buf, long size) {
        this.remainder.returnAllocation(size);
        if (ENABLE_ACCOUNTING && buf != null) {
            DebugStackTrace dst = (DebugStackTrace)this.buffers.get(buf);
            if (dst == null) {
                throw new IllegalStateException("Partially releasing a buffer that has already been released. Buffer: " + buf);
            }
            dst.size -= size;
            if (dst.size < 0L) {
                throw new IllegalStateException("Partially releasing a buffer that has already been released. Buffer: " + buf);
            }
        }
    }

    public void release(DrillBuf buf, long size) {
        this.remainder.returnAllocation(size);
        if (ENABLE_ACCOUNTING && buf != null && this.buffers.remove(buf) == null) {
            throw new IllegalStateException("Releasing a buffer that has already been released. Buffer: " + buf);
        }
    }

    public void close() {
        if (ENABLE_ACCOUNTING && !this.buffers.isEmpty()) {
            StringBuffer sb = new StringBuffer();
            sb.append("Attempted to close accountor with ");
            sb.append(this.buffers.size());
            sb.append(" buffer(s) still allocated");
            if (this.handle != null) {
                sb.append("for QueryId: ");
                sb.append(QueryIdHelper.getQueryId(this.handle.getQueryId()));
                sb.append(", MajorFragmentId: ");
                sb.append(this.handle.getMajorFragmentId());
                sb.append(", MinorFragmentId: ");
                sb.append(this.handle.getMinorFragmentId());
            }
            sb.append(".\n");
            LinkedListMultimap<DebugStackTrace, DebugStackTrace> multi = LinkedListMultimap.create();
            for (DebugStackTrace t : this.buffers.values()) {
                multi.put(t, t);
            }
            for (DebugStackTrace entry : multi.keySet()) {
                Collection allocs = multi.get(entry);
                sb.append("\n\n\tTotal ");
                sb.append(allocs.size());
                sb.append(" allocation(s) of byte size(s): ");
                for (DebugStackTrace alloc : allocs) {
                    sb.append(alloc.size);
                    sb.append(", ");
                }
                sb.append("at stack location:\n");
                entry.addToString(sb);
            }
            IllegalStateException e = new IllegalStateException(sb.toString());
            if (this.errorOnLeak) {
                throw e;
            }
            logger.warn("Memory leaked.", e);
        }
        this.remainder.close();
    }

    public class DebugStackTrace {
        private StackTraceElement[] elements;
        private long size;

        public DebugStackTrace(long size, StackTraceElement[] elements) {
            this.elements = elements;
            this.size = size;
        }

        public void addToString(StringBuffer sb) {
            for (int i = 3; i < this.elements.length; ++i) {
                sb.append("\t\t");
                sb.append(this.elements[i]);
                sb.append("\n");
            }
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + Arrays.hashCode(this.elements);
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            DebugStackTrace other = (DebugStackTrace)obj;
            return Arrays.equals(this.elements, other.elements);
        }
    }
}

